﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.ServiceModel;
using System.Xml.Linq;
using VA.PPMS.CRM.Plugins.Helper;

namespace VA.PPMS.CRM.Plugins
{
    public class BatchUpdate : IPlugin
    {
        private ITracingService _tracingService;
        private IOrganizationService _service;

        private const string PluginName = "BatchUpdate";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            _tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                _tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity entity = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (entity.LogicalName != "ppms_batch")
                    return;

                _tracingService.Trace("Entity found");

                // Verify status scenario
                var status = entity.GetAttributeValue<OptionSetValue>("statuscode");
                if (status != null && status.Value != (int)PpmsHelper.Batch_StatusCode.ValidationComplete)
                {
                    _tracingService.Trace("Target status not set");
                    return;
                }

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                _service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    _tracingService.Trace("Submit to queue...");
                    FabricHelper.SubmitToQueue(entity.Id.ToString());

                    // Save response XML for batch records
                    _tracingService.Trace("Saving response to batch...");
                    SaveResponse(entity);
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    _tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    _tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            _tracingService.Trace("Done");
        }

        private EntityCollection GetResults(IOrganizationService service, string detailId)
        {
            if (String.IsNullOrEmpty(detailId))
                return null;

            FilterExpression filter = new FilterExpression();
            filter.AddCondition("ppms_batchdetail", ConditionOperator.Equal, detailId);

            QueryExpression query = new QueryExpression("ppms_batchdetailresult");
            query.ColumnSet.AddColumns("ppms_batchdetailresultid", "ppms_name", "ppms_entitytype", "ppms_isvalid", "ppms_correlationid", "ppms_result", "ppms_message");
            query.Criteria.AddFilter(filter);

            return service.RetrieveMultiple(query);
        }

        private EntityCollection GetDetails(IOrganizationService service, string batchId)
        {
            if (String.IsNullOrEmpty(batchId))
                return null;

            FilterExpression filter = new FilterExpression();
            filter.AddCondition("ppms_batch", ConditionOperator.Equal, batchId);

            QueryExpression query = new QueryExpression("ppms_batchdetail");
            query.ColumnSet.AddColumns("ppms_batchdetailid", "ppms_name", "ppms_isvalid", "ppms_provider", "ppms_providerid", "ppms_transactiontype");
            query.Criteria.AddFilter(filter);

            return service.RetrieveMultiple(query);
        }

        private void SaveResponse(Entity entity)
        {
            // retrieve detail records
            _tracingService.Trace("Retrieve details");
            var details = GetDetails(_service, entity.Id.ToString());
            if (details == null || details.Entities.Count == 0)
            {
                _tracingService.Trace("Details not found");
                return;
            }
            _tracingService.Trace("Details found");

            // Create XML doc
            var doc = new XDocument();
            var root = new XElement("providers");
            XElement provider;
            XElement resultMsg;
            EntityCollection results;
            bool isValid;

            // Capture batch details
            foreach (var detail in details.Entities)
            {
                // Provider node
                //tracingService.Trace("Processing: ", detail.Id);
                provider = new XElement(
                    "provider",
                    new XElement("ProviderId", detail.GetAttributeValue<string>("ppms_providerid")),
                    new XElement("Success", detail.GetAttributeValue<bool>("ppms_isvalid"))
                );

                // Retrieve batch detail results
                //tracingService.Trace("Retrieve results");
                results = GetResults(_service, detail.Id.ToString());

                // Capture batch detail results
                if (results != null && results.Entities.Count > 0)
                {
                    foreach (var result in results.Entities)
                    {
                        isValid = result.GetAttributeValue<bool>("ppms_isvalid");
                        resultMsg = new XElement(
                            result.GetAttributeValue<string>("ppms_entitytype"),
                            new XElement("Id", result.GetAttributeValue<string>("ppms_name")),
                            new XElement("Success", result.GetAttributeValue<bool>("ppms_isvalid"))
                        );

                        if (!isValid)
                        {
                            resultMsg.Add(
                                new XElement("Result", result.GetAttributeValue<string>("ppms_result")),
                                new XElement("Message", result.GetAttributeValue<string>("ppms_message"))
                            );
                        }

                        provider.Add(new XElement("Results", resultMsg));
                    }
                }
                else
                {
                    _tracingService.Trace("Results not found {0}", detail.GetAttributeValue<string>("ppms_name"));
                }
                //Add to list of providers
                root.Add(provider);
            }

            // Add to document
            doc.Add(root);
            //doc.Save("c:\\dev\\output\\batch.xml");
            entity.Attributes["ppms_response"] = doc.ToString();
            _tracingService.Trace("Save complete");
        }
    }
}
